package easik.sketch.path;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;

import easik.Easik;
import easik.sketch.edge.SketchEdge;
import easik.sketch.vertex.EntityNode;


/**
 * This class is used to track path data used for constraints
 * 
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @since 2006-05-16 Kevin Green
 * @version 2006-07-04 Kevin Green
 */
public class SketchPath {
	
	/**
	 * The unique id of this path
	 */
	private String _id;
	/**
	 * The list of edges comprising this path
	 */
	private LinkedList<SketchEdge> _edges;
	/**
	 * The domain of this path
	 */
	private EntityNode _domain;
	/**
	 * The codomain of this path
	 */
	private EntityNode _coDomain;
	
	/**
	 * Default constructor takes one entity
	 * 
	 * @param inEntity An entity
	 */
	public SketchPath(EntityNode inEntity){
		_domain = inEntity;
		_coDomain = inEntity;
		_edges = new LinkedList<SketchEdge>();
		this.generateID();
	}
	
	/**
	 * Default constructor takes a LinkedList of edges
	 * 
	 * @param inPath A LinkedList of edges
	 */
	public SketchPath(LinkedList<SketchEdge> inPath){
		_edges = inPath;
		_domain = (EntityNode)((SketchEdge)inPath.getFirst()).getSourceObj();
		_coDomain = (EntityNode)((SketchEdge)inPath.getLast()).getTargetObj();
		this.generateID();
	}
	
	/**
	 * Generates a new ID based on the edges in the path
	 *
	 */
	public void generateID(){
		if(_edges.size() == 0){
			_id = _domain.getName() + "_identity";
		}
		else{
			_id = "";
			for(int i=0; i<_edges.size(); i++){
				_id += ((SketchEdge)_edges.get(i)).getName() + ";";
			}
			_id = _id.substring(0,(_id.length()-1));
		}
	}
	
	/**
	 * Overloaded method returns the _id
	 * @return The id of this path
	 */
	public String toString(){
		return getId();
	}
	
	/**
	 * Returns the domain of the path
	 * 
	 * @return The domain of the path
	 */
	public EntityNode getDomain(){
		return _domain;
	}
	
	/**
	 * Returns the ID of the path
	 * 
	 * @return The ID of the path
	 */
	public String getId(){
		generateID();
		return _id;
	}
	
	/**
	 * Returns the list of edges
	 * 
	 * @return The list of edges
	 */
	public LinkedList<SketchEdge> getEdges(){
		return _edges;
	}
	
	/**
	 * Returns the codomain of the path
	 * 
	 * @return The codomain of the path
	 */
	public EntityNode getCoDomain(){
		return _coDomain;
	}

	/**
	 * Static method to rename a path in instances where an edge contained within 
	 * the path has been renamed.
	 * 
	 * @param paths hashMap of all paths to be searched for edge.
	 * @param newEdge The new name of the path
	 * @since 2006-05-29 Vera Ranieri
	 */
	public static void renamePaths(HashMap<String, SketchPath> paths, SketchEdge newEdge){
		//Must use an array because you cannot change the thing you are iterating on
		ArrayList<SketchPath> updtPaths = new ArrayList<SketchPath>();
		
		//Iterate through paths to see which ones are changed
		Iterator it = paths.keySet().iterator();
		while (it.hasNext()){
			String key = (String) it.next();
			SketchPath p = paths.get(key);
			
			if(p.getEdges().contains(newEdge)){
				updtPaths.add(p);
			}
		}
		
		//Update changed paths after iteration
		while(updtPaths.size()>0){
			SketchPath p = (SketchPath) updtPaths.get(0);
			paths.remove(p);
			p.generateID();
			paths.put(p.getId(), p);
			updtPaths.remove(0);
		}
		Easik.getInstance().getFrame().getInfoTreeUI().refreshTree();
	}
	
	/**
	 * Method to determine whether two paths are equal, based on the edges that comprise the path.  This determination is
	 * based on the id of the path, which in turn is based on the edges involved in the path.
	 * 
	 * @param a The first path
	 * @param b The second path
	 * @return True if the paths are equal, false otherwise
	 */
	public static boolean pathsAreEqual(SketchPath a, SketchPath b){
		return (a.getId().equals(b.getId()));
	}
}
